﻿using System.IO;
using System.Net;
using System.Text;
using System.Threading;

public static class HttpWebRequestExtensions
{
    private const string RETRY_INDICATOR = "RETRY";

    public static string GetResponseWithExponentialBackoff(this WebRequest request, string requestUrl, int maxRetries = 6)
    {
        var response = GetResponse(request, requestUrl);
        if (response != null && Retry(response))
        {
            var retryCount = 0;
            while (++retryCount <= maxRetries && !Retry(response))
            {
                var sleepTimeMs = (2 ^ retryCount) * 1000;

                Thread.Sleep(sleepTimeMs);

                response = GetResponse(request, requestUrl);
            }

            if (retryCount == maxRetries)
                throw new WebException(string.Format("Timeout with {0} retries.", retryCount));
        }

        return response;
    }

    internal static string GetResponse(WebRequest request, string requestUrl)
    {
        const int MAX_READ_CHARS = 500000;

        string result = null;

        using (var response = request.GetResponse() as HttpWebResponse)
        {
            if (response == null)
                throw new WebException(string.Format("{0}: Response is null", requestUrl));

            if (response.StatusCode != HttpStatusCode.OK)
                throw new WebException(string.Format("{0}: request failed", requestUrl));

            var responseStream = response.GetResponseStream();
            if (responseStream == null)
                throw new WebException(string.Format("{0}: ResponseStream is null", requestUrl));

            if (response.StatusCode == HttpStatusCode.RequestTimeout)
                return RETRY_INDICATOR;

            using (var reader = new StreamReader(responseStream))
            {
                var count = 0;
                var responseBuilder = new StringBuilder();

                while (!reader.EndOfStream && count++ <= MAX_READ_CHARS)
                    responseBuilder.Append((char)reader.Read());

                result = responseBuilder.ToString();
            }
        }

        return result;
    }


    private static bool Retry(string response)
    {
        return response.Equals(RETRY_INDICATOR, System.StringComparison.OrdinalIgnoreCase);
    }
}